<?php
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../functions.php';

$token = $_GET['token'] ?? '';
$session = getActiveQRSession();

$faculties = getFaculties();
$departments = getDepartments();
$levels = getLevels();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Student Attendance - <?php echo SITE_NAME; ?></title>
    <script src="https://unpkg.com/html5-qrcode@2.3.8/html5-qrcode.min.js"></script>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: system-ui, -apple-system, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        .container {
            width: 100%;
            max-width: 500px;
        }
        .card {
            background: white;
            border-radius: 16px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.2);
            padding: 2rem;
            text-align: center;
        }
        .logo {
            font-size: 3rem;
            margin-bottom: 0.5rem;
        }
        .title {
            font-size: 1.5rem;
            font-weight: 700;
            color: #4F46E5;
            margin-bottom: 0.5rem;
        }
        .subtitle {
            color: #6B7280;
            margin-bottom: 1.5rem;
        }
        #reader {
            width: 100%;
            border-radius: 12px;
            overflow: hidden;
            margin-bottom: 1rem;
        }
        #reader video {
            border-radius: 12px;
        }
        .form-section {
            text-align: left;
        }
        .form-group {
            margin-bottom: 1rem;
        }
        label {
            display: block;
            font-weight: 600;
            color: #374151;
            margin-bottom: 0.5rem;
        }
        input, select {
            width: 100%;
            padding: 12px;
            border: 1px solid #E5E7EB;
            border-radius: 8px;
            font-size: 1rem;
        }
        input:focus, select:focus {
            outline: none;
            border-color: #4F46E5;
        }
        .btn {
            width: 100%;
            padding: 14px;
            background: #4F46E5;
            color: white;
            border: none;
            border-radius: 8px;
            font-size: 1.1rem;
            font-weight: 600;
            cursor: pointer;
            transition: background 0.3s;
            margin-top: 1rem;
        }
        .btn:hover { background: #4338CA; }
        .btn:disabled {
            background: #9CA3AF;
            cursor: not-allowed;
        }
        .hidden { display: none; }
        .message {
            padding: 1rem;
            border-radius: 8px;
            margin: 1rem 0;
        }
        .message.success {
            background: #D1FAE5;
            color: #065F46;
        }
        .message.error {
            background: #FEE2E2;
            color: #DC2626;
        }
        .message.warning {
            background: #FEF3C7;
            color: #92400E;
        }
        .overlay {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0,0,0,0.7);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 1000;
        }
        .overlay-content {
            background: white;
            padding: 2rem;
            border-radius: 12px;
            text-align: center;
        }
        .spinner {
            width: 50px;
            height: 50px;
            border: 4px solid #E5E7EB;
            border-top-color: #4F46E5;
            border-radius: 50%;
            animation: spin 1s linear infinite;
            margin: 0 auto 1rem;
        }
        @keyframes spin {
            to { transform: rotate(360deg); }
        }
        .session-info {
            background: #EEF2FF;
            border-radius: 8px;
            padding: 1rem;
            margin-bottom: 1rem;
            text-align: left;
            font-size: 0.9rem;
        }
        .session-info p {
            margin: 0.25rem 0;
            color: #374151;
        }
        .session-info strong {
            color: #4F46E5;
        }
        .info-text {
            font-size: 0.85rem;
            color: #6B7280;
            margin-top: 0.25rem;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="card">
            <!-- Scanner Section -->
            <div id="scannerSection">
                <div class="logo">📱</div>
                <h1 class="title">Student Attendance</h1>
                <p class="subtitle">Scan the QR code to mark your attendance</p>
                
                <div id="alreadyMarked" class="message error hidden">
                    ⚠️ You have marked attendance already.
                </div>
                
                <div id="reader"></div>
                <p class="info-text">Allow camera access to scan QR code</p>
            </div>
            
            <!-- Form Section -->
            <div id="formSection" class="hidden">
                <div class="logo">📝</div>
                <h1 class="title">Mark Attendance</h1>
                
                <div class="session-info" id="sessionInfo"></div>
                
                <form id="attendanceForm" class="form-section">
                    <div class="form-group">
                        <label>Select Faculty</label>
                        <select name="faculty" id="facultySelect" required>
                            <option value="">-- Select Faculty --</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label>Select Department</label>
                        <select name="department" id="departmentSelect" required disabled>
                            <option value="">-- Select Department --</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label>Select Level</label>
                        <select name="level" id="levelSelect" required disabled>
                            <option value="">-- Select Level --</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label>Full Name</label>
                        <input type="text" name="fullname" id="fullnameInput" required placeholder="Enter your full name">
                    </div>
                    
                    <div class="form-group">
                        <label>Matric Number</label>
                        <input type="text" name="matric" id="matricInput" required placeholder="Enter your matric number">
                    </div>
                    
                    <button type="submit" class="btn">✅ SUBMIT ATTENDANCE</button>
                </form>
            </div>
            
            <!-- Success Section -->
            <div id="successSection" class="hidden">
                <div class="logo">✅</div>
                <h1 class="title" style="color: #10B981;">Success!</h1>
                <div class="message success">
                    YOU HAVE SUBMITTED THIS ATTENDANCE SUCCESSFULLY
                    <br><br>
                    <strong>Note:</strong> You can't submit attendance again.
                </div>
            </div>
            
            <!-- Error Section -->
            <div id="errorSection" class="hidden">
                <div class="logo">❌</div>
                <h1 class="title" style="color: #EF4444;">Error</h1>
                <div class="message error" id="errorMessage"></div>
                <button class="btn" onclick="location.reload()">Try Again</button>
            </div>
        </div>
    </div>
    
    <div class="overlay hidden" id="loadingOverlay">
        <div class="overlay-content">
            <div class="spinner"></div>
            <p>Submitting attendance...</p>
        </div>
    </div>
    
    <script>
        // Data from PHP
        const allFaculties = <?php echo json_encode($faculties); ?>;
        const allDepartments = <?php echo json_encode($departments); ?>;
        const allLevels = <?php echo json_encode($levels); ?>;
        
        let scannedToken = '<?php echo $token; ?>';
        let sessionData = null;
        let html5QrCode = null;
        
        // Device fingerprint for preventing duplicate submissions
        function getDeviceFingerprint() {
            const canvas = document.createElement('canvas');
            const ctx = canvas.getContext('2d');
            ctx.textBaseline = 'top';
            ctx.font = '14px Arial';
            ctx.fillText('fingerprint', 2, 2);
            const fingerprint = canvas.toDataURL();
            
            const data = [
                navigator.userAgent,
                navigator.language,
                screen.width + 'x' + screen.height,
                new Date().getTimezoneOffset(),
                fingerprint
            ].join('|');
            
            // Simple hash
            let hash = 0;
            for (let i = 0; i < data.length; i++) {
                const char = data.charCodeAt(i);
                hash = ((hash << 5) - hash) + char;
                hash = hash & hash;
            }
            return Math.abs(hash).toString(36);
        }
        
        // Check if already marked attendance
        function hasMarkedAttendance() {
            const fingerprint = getDeviceFingerprint();
            const marked = localStorage.getItem('attendance_marked_' + fingerprint);
            const sessionMarked = sessionStorage.getItem('attendance_submitted');
            return marked || sessionMarked;
        }
        
        function setMarkedAttendance() {
            const fingerprint = getDeviceFingerprint();
            localStorage.setItem('attendance_marked_' + fingerprint, Date.now());
            sessionStorage.setItem('attendance_submitted', 'true');
            
            // Also set a cookie for extra protection
            document.cookie = 'attendance_marked=' + fingerprint + '; max-age=86400; path=/';
        }
        
        // Initialize
        document.addEventListener('DOMContentLoaded', function() {
            // Check if already marked
            if (hasMarkedAttendance()) {
                document.getElementById('alreadyMarked').classList.remove('hidden');
                return;
            }
            
            // If token provided in URL, validate it
            if (scannedToken) {
                validateToken(scannedToken);
            } else {
                initScanner();
            }
        });
        
        function initScanner() {
            html5QrCode = new Html5Qrcode("reader");
            
            Html5Qrcode.getCameras().then(devices => {
                if (devices && devices.length) {
                    const cameraId = devices[devices.length - 1].id; // Prefer back camera
                    
                    html5QrCode.start(
                        cameraId,
                        {
                            fps: 10,
                            qrbox: { width: 250, height: 250 }
                        },
                        onScanSuccess,
                        onScanFailure
                    ).catch(err => {
                        console.error('Camera start error:', err);
                        showError('Could not access camera. Please grant camera permission and refresh.');
                    });
                }
            }).catch(err => {
                console.error('Camera enumeration error:', err);
                showError('Could not access camera. Please grant camera permission.');
            });
        }
        
        function onScanSuccess(decodedText, decodedResult) {
            // Stop scanner
            if (html5QrCode) {
                html5QrCode.stop().catch(console.error);
            }
            
            // Extract token from URL
            try {
                const url = new URL(decodedText);
                const token = url.searchParams.get('token');
                if (token) {
                    validateToken(token);
                } else {
                    showError('Invalid QR code');
                }
            } catch (e) {
                showError('Invalid QR code format');
            }
        }
        
        function onScanFailure(error) {
            // Ignore scan failures (normal during scanning)
        }
        
        function validateToken(token) {
            fetch('<?php echo getBaseUrl(); ?>/api/qr-status', {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                body: 'action=validate_token&token=' + encodeURIComponent(token)
            })
            .then(res => res.json())
            .then(data => {
                if (data.success) {
                    scannedToken = token;
                    sessionData = data.session;
                    showForm();
                } else {
                    showError(data.message || 'QR code used, scan another');
                }
            })
            .catch(err => {
                showError('Connection error. Please try again.');
            });
        }
        
        function showForm() {
            document.getElementById('scannerSection').classList.add('hidden');
            document.getElementById('formSection').classList.remove('hidden');
            
            // Display session info
            const infoHtml = `
                <p><strong>Course:</strong> ${sessionData.course}</p>
                <p><strong>Lecturer:</strong> ${sessionData.lecturer_name}</p>
            `;
            document.getElementById('sessionInfo').innerHTML = infoHtml;
            
            // Populate faculty dropdown with session faculties
            const facultySelect = document.getElementById('facultySelect');
            sessionData.faculties.forEach(facultyName => {
                const faculty = allFaculties.find(f => f.name === facultyName);
                if (faculty) {
                    const option = document.createElement('option');
                    option.value = faculty.id;
                    option.textContent = faculty.name;
                    option.dataset.name = faculty.name;
                    facultySelect.appendChild(option);
                }
            });
            
            // Faculty change handler
            facultySelect.addEventListener('change', function() {
                const departmentSelect = document.getElementById('departmentSelect');
                const levelSelect = document.getElementById('levelSelect');
                
                departmentSelect.innerHTML = '<option value="">-- Select Department --</option>';
                levelSelect.innerHTML = '<option value="">-- Select Level --</option>';
                departmentSelect.disabled = true;
                levelSelect.disabled = true;
                
                if (this.value) {
                    // Filter departments by faculty and session departments
                    const filteredDepts = allDepartments.filter(d => 
                        d.faculty_id === this.value && 
                        sessionData.departments.includes(d.name)
                    );
                    
                    filteredDepts.forEach(dept => {
                        const option = document.createElement('option');
                        option.value = dept.id;
                        option.textContent = dept.name;
                        option.dataset.name = dept.name;
                        departmentSelect.appendChild(option);
                    });
                    
                    departmentSelect.disabled = false;
                }
            });
            
            // Department change handler
            document.getElementById('departmentSelect').addEventListener('change', function() {
                const levelSelect = document.getElementById('levelSelect');
                levelSelect.innerHTML = '<option value="">-- Select Level --</option>';
                levelSelect.disabled = true;
                
                if (this.value) {
                    // Get levels for this department that are in session levels
                    const deptLevels = allLevels.filter(l => l.department_id === this.value);
                    const sessionLevelNames = sessionData.levels;
                    
                    deptLevels.forEach(level => {
                        if (sessionLevelNames.includes(level.name)) {
                            const option = document.createElement('option');
                            option.value = level.id;
                            option.textContent = level.name;
                            option.dataset.name = level.name;
                            levelSelect.appendChild(option);
                        }
                    });
                    
                    levelSelect.disabled = false;
                }
            });
        }
        
        // Form submission
        document.getElementById('attendanceForm').addEventListener('submit', function(e) {
            e.preventDefault();
            
            // Check again if already marked
            if (hasMarkedAttendance()) {
                showError('You have marked attendance already.');
                return;
            }
            
            const formData = {
                token: scannedToken,
                fullname: document.getElementById('fullnameInput').value.trim(),
                matric: document.getElementById('matricInput').value.trim(),
                department: document.getElementById('departmentSelect').options[document.getElementById('departmentSelect').selectedIndex].dataset.name,
                level: document.getElementById('levelSelect').options[document.getElementById('levelSelect').selectedIndex].dataset.name,
                fingerprint: getDeviceFingerprint()
            };
            
            // Show loading
            document.getElementById('loadingOverlay').classList.remove('hidden');
            
            fetch('<?php echo getBaseUrl(); ?>/api/submit-attendance', {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                body: Object.keys(formData).map(key => 
                    encodeURIComponent(key) + '=' + encodeURIComponent(formData[key])
                ).join('&')
            })
            .then(res => res.json())
            .then(data => {
                document.getElementById('loadingOverlay').classList.add('hidden');
                
                if (data.success) {
                    setMarkedAttendance();
                    document.getElementById('formSection').classList.add('hidden');
                    document.getElementById('successSection').classList.remove('hidden');
                } else {
                    showError(data.message || 'Failed to submit attendance');
                }
            })
            .catch(err => {
                document.getElementById('loadingOverlay').classList.add('hidden');
                showError('Connection error. Please try again.');
            });
        });
        
        function showError(message) {
            document.getElementById('scannerSection').classList.add('hidden');
            document.getElementById('formSection').classList.add('hidden');
            document.getElementById('successSection').classList.add('hidden');
            document.getElementById('errorSection').classList.remove('hidden');
            document.getElementById('errorMessage').textContent = message;
            
            if (html5QrCode) {
                html5QrCode.stop().catch(console.error);
            }
        }
    </script>
</body>
</html>