<?php
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../functions.php';

if (!isLecturerLoggedIn()) {
    redirect('lecturer/login');
}

$lecturerName = $_SESSION['lecturer_name'];

// Handle AJAX requests
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    header('Content-Type: application/json');
    $action = $_POST['action'] ?? '';
    
    if ($action === 'start_session') {
        $sessionData = json_decode($_POST['session_data'], true);
        $sessionData['lecturer_name'] = $lecturerName;
        $sessionData['lecturer_id'] = $_SESSION['lecturer_id'];
        $sessionData['started_at'] = date('Y-m-d H:i:s');
        $sessionData['qr_token'] = generateQRToken();
        $sessionData['qr_generated_at'] = time();
        $sessionData['used_tokens'] = [];
        $sessionData['attendance'] = [];
        $sessionData['status'] = 'active';
        
        saveQRSession($sessionData);
        echo json_encode(['success' => true, 'token' => $sessionData['qr_token']]);
        exit;
    }
    
    if ($action === 'refresh_qr') {
        $session = getActiveQRSession();
        if ($session && $session['status'] === 'active') {
            $session['qr_token'] = generateQRToken();
            $session['qr_generated_at'] = time();
            saveQRSession($session);
            echo json_encode(['success' => true, 'token' => $session['qr_token']]);
        } else {
            echo json_encode(['success' => false, 'message' => 'No active session']);
        }
        exit;
    }
    
    if ($action === 'get_status') {
        $session = getActiveQRSession();
        if ($session) {
            echo json_encode([
                'success' => true,
                'status' => $session['status'],
                'attendance_count' => count($session['attendance']),
                'token' => $session['qr_token'],
                'token_age' => time() - $session['qr_generated_at']
            ]);
        } else {
            echo json_encode(['success' => false]);
        }
        exit;
    }
    
    if ($action === 'end_session') {
        $session = getActiveQRSession();
        if ($session) {
            $session['status'] = 'ended';
            $session['qr_token'] = '';
            saveQRSession($session);
            echo json_encode(['success' => true, 'attendance' => $session['attendance']]);
        } else {
            echo json_encode(['success' => false]);
        }
        exit;
    }
    
    if ($action === 'save_attendance') {
        $session = getActiveQRSession();
        if ($session && count($session['attendance']) > 0) {
            // Generate filename
            $timestamp = date('Y-m-d_H-i-s');
            $faculty = str_replace(' ', '-', $session['faculties'][0] ?? 'Unknown');
            $department = str_replace(' ', '-', $session['departments'][0] ?? 'Unknown');
            $course = str_replace(' ', '-', preg_replace('/[^a-zA-Z0-9\-]/', '', $session['course'] ?? 'Unknown'));
            $lecturer = str_replace(' ', '-', $session['lecturer_name']);
            
            $filename = "{$timestamp}_{$faculty}_{$department}_{$course}_{$lecturer}.csv";
            
            writeAttendanceCSV($filename, $session['attendance']);
            clearQRSession();
            
            echo json_encode(['success' => true, 'filename' => $filename]);
        } else {
            echo json_encode(['success' => false, 'message' => 'No attendance to save']);
        }
        exit;
    }
    
    if ($action === 'terminate_session') {
        clearQRSession();
        echo json_encode(['success' => true]);
        exit;
    }
    
    exit;
}

// Check for existing session
$existingSession = getActiveQRSession();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>QR Code Attendance - <?php echo SITE_NAME; ?></title>
    <script src="https://cdn.jsdelivr.net/npm/qrcode@1.5.3/build/qrcode.min.js"></script>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: system-ui, -apple-system, sans-serif;
            background: #F9FAFB;
            min-height: 100vh;
        }
        .navbar {
            background: white;
            padding: 1rem 2rem;
            box-shadow: 0 2px 4px rgba(0,0,0,0.05);
            display: flex;
            justify-content: space-between;
            align-items: center;
            flex-wrap: wrap;
            gap: 1rem;
        }
        .navbar-brand {
            font-size: 1.25rem;
            font-weight: 700;
            color: #4F46E5;
            text-decoration: none;
        }
        .navbar-nav {
            display: flex;
            gap: 1rem;
            flex-wrap: wrap;
        }
        .nav-link {
            color: #6B7280;
            text-decoration: none;
            padding: 0.5rem 1rem;
            border-radius: 6px;
            transition: all 0.3s;
        }
        .nav-link:hover, .nav-link.active {
            background: #EEF2FF;
            color: #4F46E5;
        }
        .nav-link.logout {
            background: #FEE2E2;
            color: #DC2626;
        }
        .container {
            max-width: 600px;
            margin: 0 auto;
            padding: 2rem;
        }
        .page-title {
            font-size: 1.75rem;
            font-weight: 700;
            color: #111827;
            margin-bottom: 1rem;
            text-align: center;
        }
        .card {
            background: white;
            border-radius: 12px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.05);
            padding: 2rem;
            text-align: center;
        }
        .session-info {
            background: #EEF2FF;
            border-radius: 8px;
            padding: 1rem;
            margin-bottom: 1.5rem;
            text-align: left;
        }
        .session-info p {
            margin: 0.25rem 0;
            color: #374151;
            font-size: 0.9rem;
        }
        .session-info strong {
            color: #4F46E5;
        }
        .qr-container {
            background: white;
            border: 2px solid #E5E7EB;
            border-radius: 12px;
            padding: 1.5rem;
            margin: 1.5rem 0;
            display: inline-block;
        }
        #qrcode {
            width: 250px;
            height: 250px;
            margin: 0 auto;
        }
        #qrcode canvas {
            width: 100% !important;
            height: 100% !important;
        }
        .qr-blank {
            width: 250px;
            height: 250px;
            background: #F3F4F6;
            display: flex;
            align-items: center;
            justify-content: center;
            color: #9CA3AF;
            font-size: 1.1rem;
        }
        .timer {
            font-size: 1.5rem;
            font-weight: 700;
            color: #4F46E5;
            margin: 1rem 0;
        }
        .attendance-count {
            font-size: 1.25rem;
            color: #10B981;
            margin: 1rem 0;
        }
        .btn-group {
            display: flex;
            gap: 1rem;
            justify-content: center;
            flex-wrap: wrap;
            margin-top: 1.5rem;
        }
        .btn {
            padding: 12px 24px;
            border: none;
            border-radius: 8px;
            font-size: 1rem;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s;
        }
        .btn-primary {
            background: #4F46E5;
            color: white;
        }
        .btn-primary:hover { background: #4338CA; }
        .btn-success {
            background: #10B981;
            color: white;
        }
        .btn-success:hover { background: #059669; }
        .btn-danger {
            background: #EF4444;
            color: white;
        }
        .btn-danger:hover { background: #DC2626; }
        .btn-warning {
            background: #F59E0B;
            color: white;
        }
        .btn-warning:hover { background: #D97706; }
        .hidden { display: none; }
        .status-badge {
            display: inline-block;
            padding: 0.5rem 1rem;
            border-radius: 20px;
            font-weight: 600;
            margin-bottom: 1rem;
        }
        .status-active {
            background: #D1FAE5;
            color: #065F46;
        }
        .status-ended {
            background: #FEE2E2;
            color: #DC2626;
        }
        .no-session {
            padding: 2rem;
            color: #6B7280;
        }
        .no-session a {
            color: #4F46E5;
            text-decoration: none;
        }
        @media (max-width: 768px) {
            .navbar { padding: 1rem; }
            .container { padding: 1rem; }
            .btn { padding: 10px 16px; font-size: 0.9rem; }
        }
    </style>
</head>
<body>
    <nav class="navbar">
        <a href="dashboard" class="navbar-brand">👨‍🏫 Lecturer Portal</a>
        <div class="navbar-nav">
            <a href="dashboard" class="nav-link">Dashboard</a>
            <a href="assign" class="nav-link">Take Attendance</a>
            <a href="attendance-list" class="nav-link">Attendance List</a>
            <a href="logout" class="nav-link logout">Logout</a>
        </div>
    </nav>
    
    <div class="container">
        <h1 class="page-title">📱 QR Code Attendance</h1>
        
        <div class="card">
            <div id="noSession" class="no-session">
                <p>No attendance session started.</p>
                <p><a href="assign">← Go back to start a new attendance</a></p>
            </div>
            
            <div id="activeSession" class="hidden">
                <span class="status-badge status-active" id="statusBadge">🟢 Active</span>
                
                <div class="session-info" id="sessionInfo"></div>
                
                <div class="qr-container">
                    <div id="qrcode"></div>
                    <div id="qrBlank" class="qr-blank hidden">Attendance Ended</div>
                </div>
                
                <div class="timer" id="timer">5s</div>
                <div class="attendance-count" id="attendanceCount">Students: 0</div>
                
                <div class="btn-group" id="activeButtons">
                    <button class="btn btn-danger" onclick="endSession()">🛑 END</button>
                    <button class="btn btn-warning" onclick="terminateSession()">❌ TERMINATE</button>
                </div>
                
                <div class="btn-group hidden" id="endedButtons">
                    <button class="btn btn-success" onclick="saveAttendance()">💾 SAVE ATTENDANCE</button>
                    <button class="btn btn-warning" onclick="terminateSession()">❌ DISCARD</button>
                </div>
            </div>
        </div>
    </div>
    
    <script>
        let sessionData = null;
        let timerInterval = null;
        let refreshInterval = null;
        let currentToken = '';
        let isEnded = false;
        
        // Check for session data from assign page
        const storedSession = sessionStorage.getItem('attendanceSession');
        
        // Check for existing server session
        <?php if ($existingSession && $existingSession['status'] === 'active'): ?>
        sessionData = <?php echo json_encode($existingSession); ?>;
        currentToken = sessionData.qr_token;
        initializeSession();
        <?php elseif ($storedSession): ?>
        // Start new session from stored data
        document.addEventListener('DOMContentLoaded', function() {
            if (storedSession) {
                sessionData = JSON.parse(storedSession);
                sessionStorage.removeItem('attendanceSession');
                startNewSession();
            }
        });
        <?php endif; ?>
        
        function startNewSession() {
            fetch('qrcode', {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                body: 'action=start_session&session_data=' + encodeURIComponent(JSON.stringify(sessionData))
            })
            .then(res => res.json())
            .then(data => {
                if (data.success) {
                    currentToken = data.token;
                    initializeSession();
                }
            });
        }
        
        function initializeSession() {
            document.getElementById('noSession').classList.add('hidden');
            document.getElementById('activeSession').classList.remove('hidden');
            
            // Display session info
            const infoHtml = `
                <p><strong>Faculties:</strong> ${sessionData.faculties.join(', ')}</p>
                <p><strong>Departments:</strong> ${sessionData.departments.join(', ')}</p>
                <p><strong>Levels:</strong> ${sessionData.levels.join(', ')}</p>
                <p><strong>Course:</strong> ${sessionData.course}</p>
            `;
            document.getElementById('sessionInfo').innerHTML = infoHtml;
            
            generateQRCode(currentToken);
            startTimer();
            startStatusPolling();
        }
        
        function generateQRCode(token) {
            const qrElement = document.getElementById('qrcode');
            qrElement.innerHTML = '';
            
            const baseUrl = window.location.origin + window.location.pathname.replace('/lecturer/qrcode', '');
            const scanUrl = baseUrl + '/studentattendance?token=' + token;
            
            QRCode.toCanvas(document.createElement('canvas'), scanUrl, {
                width: 250,
                margin: 2,
                color: { dark: '#4F46E5', light: '#FFFFFF' }
            }, function(error, canvas) {
                if (!error) {
                    qrElement.appendChild(canvas);
                }
            });
        }
        
        function startTimer() {
            let seconds = 5;
            const timerEl = document.getElementById('timer');
            
            if (timerInterval) clearInterval(timerInterval);
            
            timerInterval = setInterval(() => {
                seconds--;
                timerEl.textContent = seconds + 's';
                
                if (seconds <= 0) {
                    refreshQRCode();
                    seconds = 5;
                }
            }, 1000);
        }
        
        function refreshQRCode() {
            if (isEnded) return;
            
            fetch('qrcode', {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                body: 'action=refresh_qr'
            })
            .then(res => res.json())
            .then(data => {
                if (data.success) {
                    currentToken = data.token;
                    generateQRCode(currentToken);
                }
            });
        }
        
        function startStatusPolling() {
            if (refreshInterval) clearInterval(refreshInterval);
            
            refreshInterval = setInterval(() => {
                fetch('qrcode', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                    body: 'action=get_status'
                })
                .then(res => res.json())
                .then(data => {
                    if (data.success) {
                        document.getElementById('attendanceCount').textContent = 'Students: ' + data.attendance_count;
                        
                        // Check if token was used (someone scanned)
                        if (data.token !== currentToken && !isEnded) {
                            currentToken = data.token;
                            generateQRCode(currentToken);
                        }
                    }
                });
            }, 1000);
        }
        
        function endSession() {
            if (!confirm('End this attendance session?')) return;
            
            isEnded = true;
            if (timerInterval) clearInterval(timerInterval);
            
            fetch('qrcode', {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                body: 'action=end_session'
            })
            .then(res => res.json())
            .then(data => {
                if (data.success) {
                    document.getElementById('qrcode').classList.add('hidden');
                    document.getElementById('qrBlank').classList.remove('hidden');
                    document.getElementById('timer').textContent = 'Ended';
                    document.getElementById('statusBadge').className = 'status-badge status-ended';
                    document.getElementById('statusBadge').textContent = '🔴 Ended';
                    document.getElementById('activeButtons').classList.add('hidden');
                    document.getElementById('endedButtons').classList.remove('hidden');
                    document.getElementById('attendanceCount').textContent = 'Total Students: ' + data.attendance.length;
                }
            });
        }
        
        function saveAttendance() {
            fetch('qrcode', {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                body: 'action=save_attendance'
            })
            .then(res => res.json())
            .then(data => {
                if (data.success) {
                    alert('Attendance saved successfully!\nFile: ' + data.filename);
                    window.location.href = 'attendance-list';
                } else {
                    alert(data.message || 'Failed to save attendance');
                }
            });
        }
        
        function terminateSession() {
            if (!confirm('Terminate and discard this attendance session?')) return;
            
            if (timerInterval) clearInterval(timerInterval);
            if (refreshInterval) clearInterval(refreshInterval);
            
            fetch('qrcode', {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                body: 'action=terminate_session'
            })
            .then(res => res.json())
            .then(data => {
                if (data.success) {
                    window.location.href = 'assign';
                }
            });
        }
        
        // Cleanup on page unload
        window.addEventListener('beforeunload', function() {
            if (timerInterval) clearInterval(timerInterval);
            if (refreshInterval) clearInterval(refreshInterval);
        });
    </script>
</body>
</html>