<?php
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../functions.php';

if (!isLecturerLoggedIn()) {
    redirect('lecturer/login');
}

$faculties = getFaculties();
$departments = getDepartments();
$levels = getLevels();
$courses = getCourses();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Assign Attendance - <?php echo SITE_NAME; ?></title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: system-ui, -apple-system, sans-serif;
            background: #F9FAFB;
            min-height: 100vh;
        }
        .navbar {
            background: white;
            padding: 1rem 2rem;
            box-shadow: 0 2px 4px rgba(0,0,0,0.05);
            display: flex;
            justify-content: space-between;
            align-items: center;
            flex-wrap: wrap;
            gap: 1rem;
        }
        .navbar-brand {
            font-size: 1.25rem;
            font-weight: 700;
            color: #4F46E5;
            text-decoration: none;
        }
        .navbar-nav {
            display: flex;
            gap: 1rem;
            flex-wrap: wrap;
        }
        .nav-link {
            color: #6B7280;
            text-decoration: none;
            padding: 0.5rem 1rem;
            border-radius: 6px;
            transition: all 0.3s;
        }
        .nav-link:hover, .nav-link.active {
            background: #EEF2FF;
            color: #4F46E5;
        }
        .nav-link.logout {
            background: #FEE2E2;
            color: #DC2626;
        }
        .container {
            max-width: 800px;
            margin: 0 auto;
            padding: 2rem;
        }
        .page-title {
            font-size: 1.75rem;
            font-weight: 700;
            color: #111827;
            margin-bottom: 2rem;
        }
        .card {
            background: white;
            border-radius: 12px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.05);
            padding: 1.5rem;
        }
        .form-group {
            margin-bottom: 1.5rem;
        }
        label {
            display: block;
            font-weight: 600;
            color: #374151;
            margin-bottom: 0.5rem;
        }
        .checkbox-group {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(180px, 1fr));
            gap: 0.5rem;
            max-height: 200px;
            overflow-y: auto;
            padding: 0.5rem;
            border: 1px solid #E5E7EB;
            border-radius: 8px;
        }
        .checkbox-item {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            padding: 0.5rem;
            background: #F9FAFB;
            border-radius: 6px;
            cursor: pointer;
            transition: background 0.2s;
        }
        .checkbox-item:hover {
            background: #EEF2FF;
        }
        .checkbox-item input {
            width: 18px;
            height: 18px;
            cursor: pointer;
        }
        select {
            width: 100%;
            padding: 12px;
            border: 1px solid #E5E7EB;
            border-radius: 8px;
            font-size: 1rem;
            background: white;
        }
        select:focus {
            outline: none;
            border-color: #4F46E5;
        }
        .btn {
            width: 100%;
            padding: 14px;
            background: #4F46E5;
            color: white;
            border: none;
            border-radius: 8px;
            font-size: 1.1rem;
            font-weight: 600;
            cursor: pointer;
            transition: background 0.3s;
        }
        .btn:hover { background: #4338CA; }
        .btn:disabled {
            background: #9CA3AF;
            cursor: not-allowed;
        }
        .hidden { display: none; }
        .overlay {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0,0,0,0.5);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 1000;
        }
        .overlay-content {
            background: white;
            padding: 2rem;
            border-radius: 12px;
            text-align: center;
        }
        .spinner {
            width: 50px;
            height: 50px;
            border: 4px solid #E5E7EB;
            border-top-color: #4F46E5;
            border-radius: 50%;
            animation: spin 1s linear infinite;
            margin: 0 auto 1rem;
        }
        @keyframes spin {
            to { transform: rotate(360deg); }
        }
        .info-text {
            color: #6B7280;
            font-size: 0.85rem;
            margin-top: 0.25rem;
        }
        @media (max-width: 768px) {
            .navbar { padding: 1rem; }
            .container { padding: 1rem; }
        }
    </style>
</head>
<body>
    <nav class="navbar">
        <a href="dashboard" class="navbar-brand">👨‍🏫 Lecturer Portal</a>
        <div class="navbar-nav">
            <a href="dashboard" class="nav-link">Dashboard</a>
            <a href="assign" class="nav-link active">Take Attendance</a>
            <a href="attendance-list" class="nav-link">Attendance List</a>
            <a href="logout" class="nav-link logout">Logout</a>
        </div>
    </nav>
    
    <div class="container">
        <h1 class="page-title">📝 Assign Attendance</h1>
        
        <div class="card">
            <form id="assignForm">
                <!-- Faculty Selection -->
                <div class="form-group">
                    <label>Select Faculty(ies)</label>
                    <p class="info-text">You can select one or more faculties</p>
                    <div class="checkbox-group" id="facultyGroup">
                        <?php if (empty($faculties)): ?>
                            <p style="padding: 1rem; color: #6B7280;">No faculties available. Ask admin to add faculties.</p>
                        <?php else: ?>
                            <?php foreach ($faculties as $faculty): ?>
                                <label class="checkbox-item">
                                    <input type="checkbox" name="faculties[]" value="<?php echo $faculty['id']; ?>" data-name="<?php echo htmlspecialchars($faculty['name']); ?>">
                                    <span><?php echo htmlspecialchars($faculty['name']); ?></span>
                                </label>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </div>
                </div>
                
                <!-- Department Selection -->
                <div class="form-group hidden" id="departmentSection">
                    <label>Select Department(s)</label>
                    <p class="info-text">Select departments from chosen faculties</p>
                    <div class="checkbox-group" id="departmentGroup"></div>
                </div>
                
                <!-- Level Selection -->
                <div class="form-group hidden" id="levelSection">
                    <label>Select Level(s)</label>
                    <p class="info-text">Levels are combined without duplicates</p>
                    <div class="checkbox-group" id="levelGroup"></div>
                </div>
                
                <!-- Course Selection -->
                <div class="form-group hidden" id="courseSection">
                    <label>Select Course</label>
                    <p class="info-text">Choose one course for this attendance</p>
                    <select name="course" id="courseSelect" required>
                        <option value="">-- Select Course --</option>
                        <?php foreach ($courses as $course): ?>
                            <option value="<?php echo $course['id']; ?>" data-name="<?php echo htmlspecialchars($course['code'] . ' - ' . $course['name']); ?>">
                                <?php echo htmlspecialchars($course['code'] . ' - ' . $course['name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <!-- Start Button -->
                <div class="form-group hidden" id="startSection">
                    <button type="submit" class="btn" id="startBtn">🚀 START ATTENDANCE</button>
                </div>
            </form>
        </div>
    </div>
    
    <div class="overlay hidden" id="loadingOverlay">
        <div class="overlay-content">
            <div class="spinner"></div>
            <p>Starting attendance session...</p>
        </div>
    </div>
    
    <script>
        // Data from PHP
        const allDepartments = <?php echo json_encode($departments); ?>;
        const allLevels = <?php echo json_encode($levels); ?>;
        
        // Elements
        const facultyCheckboxes = document.querySelectorAll('input[name="faculties[]"]');
        const departmentSection = document.getElementById('departmentSection');
        const departmentGroup = document.getElementById('departmentGroup');
        const levelSection = document.getElementById('levelSection');
        const levelGroup = document.getElementById('levelGroup');
        const courseSection = document.getElementById('courseSection');
        const courseSelect = document.getElementById('courseSelect');
        const startSection = document.getElementById('startSection');
        const assignForm = document.getElementById('assignForm');
        const loadingOverlay = document.getElementById('loadingOverlay');
        
        // Track selections
        let selectedFaculties = [];
        let selectedDepartments = [];
        let selectedLevels = [];
        
        // Faculty selection handler
        facultyCheckboxes.forEach(checkbox => {
            checkbox.addEventListener('change', updateDepartments);
        });
        
        function updateDepartments() {
            selectedFaculties = Array.from(document.querySelectorAll('input[name="faculties[]"]:checked'))
                .map(cb => cb.value);
            
            if (selectedFaculties.length === 0) {
                departmentSection.classList.add('hidden');
                levelSection.classList.add('hidden');
                courseSection.classList.add('hidden');
                startSection.classList.add('hidden');
                return;
            }
            
            // Filter departments by selected faculties
            const filteredDepts = allDepartments.filter(d => selectedFaculties.includes(d.faculty_id));
            
            departmentGroup.innerHTML = '';
            if (filteredDepts.length === 0) {
                departmentGroup.innerHTML = '<p style="padding: 1rem; color: #6B7280;">No departments available for selected faculties.</p>';
            } else {
                filteredDepts.forEach(dept => {
                    const label = document.createElement('label');
                    label.className = 'checkbox-item';
                    label.innerHTML = `
                        <input type="checkbox" name="departments[]" value="${dept.id}" data-name="${dept.name}">
                        <span>${dept.name}</span>
                    `;
                    departmentGroup.appendChild(label);
                });
                
                // Add event listeners to new checkboxes
                document.querySelectorAll('input[name="departments[]"]').forEach(cb => {
                    cb.addEventListener('change', updateLevels);
                });
            }
            
            departmentSection.classList.remove('hidden');
            levelSection.classList.add('hidden');
            courseSection.classList.add('hidden');
            startSection.classList.add('hidden');
        }
        
        function updateLevels() {
            selectedDepartments = Array.from(document.querySelectorAll('input[name="departments[]"]:checked'))
                .map(cb => cb.value);
            
            if (selectedDepartments.length === 0) {
                levelSection.classList.add('hidden');
                courseSection.classList.add('hidden');
                startSection.classList.add('hidden');
                return;
            }
            
            // Filter levels by selected departments and remove duplicates by name
            const filteredLevels = allLevels.filter(l => selectedDepartments.includes(l.department_id));
            const uniqueLevels = [];
            const levelNames = new Set();
            
            filteredLevels.forEach(level => {
                if (!levelNames.has(level.name)) {
                    levelNames.add(level.name);
                    uniqueLevels.push(level);
                }
            });
            
            // Sort levels numerically
            uniqueLevels.sort((a, b) => {
                const numA = parseInt(a.name) || 0;
                const numB = parseInt(b.name) || 0;
                return numA - numB;
            });
            
            levelGroup.innerHTML = '';
            if (uniqueLevels.length === 0) {
                levelGroup.innerHTML = '<p style="padding: 1rem; color: #6B7280;">No levels available for selected departments.</p>';
            } else {
                uniqueLevels.forEach(level => {
                    const label = document.createElement('label');
                    label.className = 'checkbox-item';
                    label.innerHTML = `
                        <input type="checkbox" name="levels[]" value="${level.id}" data-name="${level.name}">
                        <span>${level.name}</span>
                    `;
                    levelGroup.appendChild(label);
                });
                
                // Add event listeners to new checkboxes
                document.querySelectorAll('input[name="levels[]"]').forEach(cb => {
                    cb.addEventListener('change', updateCourseSection);
                });
            }
            
            levelSection.classList.remove('hidden');
            courseSection.classList.add('hidden');
            startSection.classList.add('hidden');
        }
        
        function updateCourseSection() {
            selectedLevels = Array.from(document.querySelectorAll('input[name="levels[]"]:checked'))
                .map(cb => cb.value);
            
            if (selectedLevels.length === 0) {
                courseSection.classList.add('hidden');
                startSection.classList.add('hidden');
                return;
            }
            
            courseSection.classList.remove('hidden');
            startSection.classList.add('hidden');
        }
        
        // Course selection handler
        courseSelect.addEventListener('change', function() {
            if (this.value) {
                startSection.classList.remove('hidden');
            } else {
                startSection.classList.add('hidden');
            }
        });
        
        // Form submission
        assignForm.addEventListener('submit', function(e) {
            e.preventDefault();
            
            // Gather all selected data
            const facultyNames = Array.from(document.querySelectorAll('input[name="faculties[]"]:checked'))
                .map(cb => cb.dataset.name);
            const departmentNames = Array.from(document.querySelectorAll('input[name="departments[]"]:checked'))
                .map(cb => cb.dataset.name);
            const levelNames = Array.from(document.querySelectorAll('input[name="levels[]"]:checked'))
                .map(cb => cb.dataset.name);
            const courseOption = courseSelect.options[courseSelect.selectedIndex];
            const courseName = courseOption.dataset.name;
            
            // Store session data
            const sessionData = {
                faculties: facultyNames,
                departments: departmentNames,
                levels: levelNames,
                course: courseName,
                courseId: courseSelect.value
            };
            
            // Show loading overlay
            loadingOverlay.classList.remove('hidden');
            
            // Store in session storage and redirect
            sessionStorage.setItem('attendanceSession', JSON.stringify(sessionData));
            
            setTimeout(() => {
                window.location.href = 'qrcode';
            }, 1500);
        });
    </script>
</body>
</html>