<?php
/**
 * Shared Utility Functions
 */

require_once __DIR__ . '/config.php';

/**
 * Read JSON data from file
 */
function readJsonFile($filename) {
    $filePath = DATA_PATH . '/' . $filename;
    if (file_exists($filePath)) {
        $content = file_get_contents($filePath);
        return json_decode($content, true) ?: [];
    }
    return [];
}

/**
 * Write JSON data to file
 */
function writeJsonFile($filename, $data) {
    $filePath = DATA_PATH . '/' . $filename;
    return file_put_contents($filePath, json_encode($data, JSON_PRETTY_PRINT));
}

/**
 * Hash password
 */
function hashPassword($password) {
    return password_hash($password, PASSWORD_DEFAULT);
}

/**
 * Verify password
 */
function verifyPassword($password, $hash) {
    return password_verify($password, $hash);
}

/**
 * Sanitize input
 */
function sanitize($input) {
    return htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
}

/**
 * Check if admin is logged in
 */
function isAdminLoggedIn() {
    return isset($_SESSION['admin_logged_in']) && $_SESSION['admin_logged_in'] === true;
}

/**
 * Check if lecturer is logged in
 */
function isLecturerLoggedIn() {
    return isset($_SESSION['lecturer_logged_in']) && $_SESSION['lecturer_logged_in'] === true;
}

/**
 * Redirect to URL
 */
function redirect($url) {
    header("Location: $url");
    exit;
}

/**
 * Get all lecturers
 */
function getLecturers() {
    return readJsonFile('lecturers.json');
}

/**
 * Get lecturer by email
 */
function getLecturerByEmail($email) {
    $lecturers = getLecturers();
    foreach ($lecturers as $lecturer) {
        if ($lecturer['email'] === $email) {
            return $lecturer;
        }
    }
    return null;
}

/**
 * Add new lecturer
 */
function addLecturer($fullname, $email, $password) {
    $lecturers = getLecturers();
    $lecturers[] = [
        'id' => uniqid(),
        'fullname' => $fullname,
        'email' => $email,
        'password' => hashPassword($password),
        'created_at' => date('Y-m-d H:i:s')
    ];
    return writeJsonFile('lecturers.json', $lecturers);
}

/**
 * Get faculties
 */
function getFaculties() {
    return readJsonFile('faculties.json');
}

/**
 * Add faculty
 */
function addFaculty($name) {
    $faculties = getFaculties();
    $faculties[] = [
        'id' => uniqid(),
        'name' => $name
    ];
    return writeJsonFile('faculties.json', $faculties);
}

/**
 * Delete faculty
 */
function deleteFaculty($id) {
    $faculties = getFaculties();
    $faculties = array_filter($faculties, function($f) use ($id) {
        return $f['id'] !== $id;
    });
    return writeJsonFile('faculties.json', array_values($faculties));
}

/**
 * Get departments
 */
function getDepartments() {
    return readJsonFile('departments.json');
}

/**
 * Add department
 */
function addDepartment($name, $facultyId) {
    $departments = getDepartments();
    $departments[] = [
        'id' => uniqid(),
        'name' => $name,
        'faculty_id' => $facultyId
    ];
    return writeJsonFile('departments.json', $departments);
}

/**
 * Delete department
 */
function deleteDepartment($id) {
    $departments = getDepartments();
    $departments = array_filter($departments, function($d) use ($id) {
        return $d['id'] !== $id;
    });
    return writeJsonFile('departments.json', array_values($departments));
}

/**
 * Get levels
 */
function getLevels() {
    return readJsonFile('levels.json');
}

/**
 * Add level
 */
function addLevel($name, $departmentId) {
    $levels = getLevels();
    $levels[] = [
        'id' => uniqid(),
        'name' => $name,
        'department_id' => $departmentId
    ];
    return writeJsonFile('levels.json', $levels);
}

/**
 * Delete level
 */
function deleteLevel($id) {
    $levels = getLevels();
    $levels = array_filter($levels, function($l) use ($id) {
        return $l['id'] !== $id;
    });
    return writeJsonFile('levels.json', array_values($levels));
}

/**
 * Get courses
 */
function getCourses() {
    return readJsonFile('courses.json');
}

/**
 * Add course
 */
function addCourse($name, $code) {
    $courses = getCourses();
    $courses[] = [
        'id' => uniqid(),
        'name' => $name,
        'code' => $code
    ];
    return writeJsonFile('courses.json', $courses);
}

/**
 * Delete course
 */
function deleteCourse($id) {
    $courses = getCourses();
    $courses = array_filter($courses, function($c) use ($id) {
        return $c['id'] !== $id;
    });
    return writeJsonFile('courses.json', array_values($courses));
}

/**
 * Get attendance files for a lecturer
 */
function getAttendanceFiles($lecturerName = null) {
    $files = [];
    if (is_dir(ATTENDANCE_PATH)) {
        $allFiles = scandir(ATTENDANCE_PATH);
        foreach ($allFiles as $file) {
            if (pathinfo($file, PATHINFO_EXTENSION) === 'csv') {
                if ($lecturerName === null || strpos($file, $lecturerName) !== false) {
                    $files[] = $file;
                }
            }
        }
    }
    return $files;
}

/**
 * Count total attendance records
 */
function countTotalAttendance($lecturerName = null) {
    $files = getAttendanceFiles($lecturerName);
    $total = 0;
    foreach ($files as $file) {
        $filePath = ATTENDANCE_PATH . '/' . $file;
        if (file_exists($filePath)) {
            $lines = file($filePath);
            $total += max(0, count($lines) - 1); // Exclude header
        }
    }
    return $total;
}

/**
 * Generate unique QR token
 */
function generateQRToken() {
    return bin2hex(random_bytes(16));
}

/**
 * Save active QR session
 */
function saveQRSession($sessionData) {
    $filePath = DATA_PATH . '/active_qr_session.json';
    return file_put_contents($filePath, json_encode($sessionData));
}

/**
 * Get active QR session
 */
function getActiveQRSession() {
    $filePath = DATA_PATH . '/active_qr_session.json';
    if (file_exists($filePath)) {
        return json_decode(file_get_contents($filePath), true);
    }
    return null;
}

/**
 * Clear active QR session
 */
function clearQRSession() {
    $filePath = DATA_PATH . '/active_qr_session.json';
    if (file_exists($filePath)) {
        unlink($filePath);
    }
}

/**
 * Read attendance CSV file
 */
function readAttendanceCSV($filename) {
    $filePath = ATTENDANCE_PATH . '/' . $filename;
    $records = [];
    if (file_exists($filePath)) {
        $handle = fopen($filePath, 'r');
        $header = fgetcsv($handle); // Skip header
        while (($row = fgetcsv($handle)) !== false) {
            $records[] = [
                'sn' => $row[0] ?? '',
                'fullname' => $row[1] ?? '',
                'matric' => $row[2] ?? '',
                'department' => $row[3] ?? '',
                'level' => $row[4] ?? ''
            ];
        }
        fclose($handle);
    }
    return $records;
}

/**
 * Write attendance to CSV
 */
function writeAttendanceCSV($filename, $records) {
    $filePath = ATTENDANCE_PATH . '/' . $filename;
    $handle = fopen($filePath, 'w');
    fputcsv($handle, ['S/N', 'Full Name', 'Matric Number', 'Department', 'Level']);
    foreach ($records as $index => $record) {
        fputcsv($handle, [
            $index + 1,
            $record['fullname'],
            $record['matric'],
            $record['department'],
            $record['level']
        ]);
    }
    fclose($handle);
    return true;
}

/**
 * Get base URL
 */
function getBaseUrl() {
    $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
    $host = $_SERVER['HTTP_HOST'];
    $script = dirname($_SERVER['SCRIPT_NAME']);
    return $protocol . '://' . $host . $script;
}
?>